/*
 * ALSA driver for Panasonic UniPhier series.
 * 
 * Copyright (c) 2013 Panasonic corporation.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MN2WS_PCM_UTIL_H__
#define MN2WS_PCM_UTIL_H__

#include <linux/kernel.h>
#include <linux/version.h>

//for warning
#ifdef __KERNEL__
#  define PRINTF_WARN(fmt, args...)    \
	snd_printk(KERN_WARNING "%s: " fmt, get_pcm_dev_name(), ##args)
#endif

//for notice
#ifdef __KERNEL__
#  define PRINTF_NOTE(fmt, args...)    \
	snd_printk(KERN_WARNING "%s: " fmt, get_pcm_dev_name(), ##args)
#endif

//for debug
#ifdef CONFIG_SND_SOC_MN2WS_PCM_DEBUG
#  ifdef __KERNEL__
#    define DPRINTF(fmt, args...) snd_printk(KERN_NOTICE fmt, ##args)
#  endif
#else
#  define DPRINTF(fmt, args...) no_printk(KERN_NOTICE fmt, ##args)
#endif //MN2WS_PCM_DEBUG


struct ringbuf {
	unsigned char *start;
	size_t len;
	loff_t rd, wr;
};

struct mem_mapping {
	size_t size;
	unsigned long paddr;
	void *mapping;
};


//utils
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 35)
int no_printk(const char *fmt, ...) 
	__attribute__((format(printf, 1, 2)));
#endif

int pcm_mmap(struct mem_mapping *mr, size_t size, unsigned long paddr);
int pcm_munmap(struct mem_mapping *mr);
u32 pcm_readl(const struct mem_mapping *mr, unsigned long paddr);
void pcm_writel(const struct mem_mapping *mr, u32 val, unsigned long paddr);

u32 rev32(u32 v);
void *memcpy_rev(void *dest, const void *src, size_t n);

int init_ringbuf(struct ringbuf *r, unsigned char *buf, size_t len);
loff_t get_rp_ringbuf(const struct ringbuf *r);
int set_rp_ringbuf(struct ringbuf *r, loff_t pos);
loff_t get_wp_ringbuf(const struct ringbuf *r);
int set_wp_ringbuf(struct ringbuf *r, loff_t pos);
size_t get_remain_ringbuf(const struct ringbuf *r);
size_t get_space_ringbuf(const struct ringbuf *r);
int fill_ringbuf(struct ringbuf *r, unsigned char d);
ssize_t read_ringbuf(struct ringbuf *r, void *buf, size_t count);
ssize_t write_ringbuf(struct ringbuf *r, const void *buf, size_t count);
ssize_t write_silent_ringbuf(struct ringbuf *r, size_t count);
ssize_t copy_ringbuf(struct ringbuf *dest, struct ringbuf *src, size_t count);

size_t get_remain(loff_t rd, loff_t wr, size_t len);
size_t get_remain_continuous(loff_t rd, loff_t wr, size_t len);
size_t get_space(loff_t rd, loff_t wr, size_t len);
size_t get_space_continuous(loff_t rd, loff_t wr, size_t len);

void timeradd(struct timeval *a, struct timeval *b, struct timeval *res);
void timersub(struct timeval *a, struct timeval *b, struct timeval *res);
void timerclear(struct timeval *tvp);

#endif //MN2WS_PCM_UTIL_H__
